--[[
Painted Fences by Clickety
Painted fences/rails/gates can be made of any wood plus dye, in the standard 15
colours and optionally light blue.
If ethereal is loaded, giant mushroom stalk blocks can replace wood.
Painted fences can be changed into painted mob fences and vice versa.
Painted fences, rails, gates, mobs fences and mobs tops can be repainted.

In older Minetest Game versions, the function to register fences/rails forces a
recipe can can cause a crash if "material" is nil, so output is used as material
and the recipe is cleared immediately afterwards.
In 5.3.0 and earlier, craftguide still displays cleared recipes.
]]

local S

if minetest.get_translator ~= nil then
    S = minetest.get_translator(minetest.get_current_modname())
else
    S = function(str)
        return(str)
    end
end

local modpath = minetest.get_modpath(minetest.get_current_modname()) .. DIR_DELIM
-- For mob fences and tops:
local mobsredo = minetest.get_modpath("mobs")
-- For the light blue dye:
local cropo = minetest.get_modpath("cropocalypse")
-- To make painted items from mushroom blocks:
local ethshroom = minetest.get_modpath("ethereal") and "ethereal:mushroom_trunk" or ""
-- For gates:
local gates = minetest.get_modpath("doors") and doors.register_fencegate
-- For wide gates:
local gates2 = minetest.get_modpath("gates_long")

local dyelist = {
	{ "white", "White" },
	{ "grey", "Grey" },
	{ "dark_grey", "Dark Grey" },
	{ "black", "Black" },
	{ "yellow", "Yellow" },
	{ "orange", "Orange" },
	{ "red", "Red" },
	{ "brown", "Brown" },
	{ "pink", "Pink" },
	{ "magenta", "Magenta" },
	{ "violet", "Violet" },
	{ "blue", "Blue" },
	{ "cyan", "Cyan" },
	{ "green", "Green" },
	{ "dark_green", "Dark Green" },
	{ "light_blue", "Light Blue" }
}

painted_fences = {
    version   = os.time({year = 2024, month = 7, day = 31}),
    author    = "Clickety",
    license   = "EUPL-1.2",
    path      = modpath, -- Not strictly necessary
    translate = S,       -- Needed for gates_long.lua
    dyelist   = dyelist  -- Ditto
}

-- Create nodes, recipes per dye.
local function paintem( pcolour, plabel )

	local fencename  = "painted_fences:fence_" .. pcolour
	local fencelabel = plabel .. " Fence"
	local frailname  = "painted_fences:fence_rail_" .. pcolour
	local fraillabel = plabel .. " Fence Rail"
	local fencetex   = "painted_fences_" .. pcolour .. ".png"
	local railtex    = "painted_fences_rail_" .. pcolour .. ".png"
	local dyename    = "dye:" .. pcolour
	if cropo and pcolour == "light_blue" then
		dyename = "cropocalypse:light_blue_dye"
	end

	-- Register painted fence
	default.register_fence(fencename, {
		description = S(fencelabel),
		texture = railtex, -- For inventory image.
--		material = nil, -- Do NOT add crafting recipe here.
		material = fencename, -- Backward compatibility
		groups = {choppy = 2, oddly_breakable_by_hand = 2, flammable = 2, paint_fence = 1 },
		sounds = default.node_sound_wood_defaults(),
	})
	-- After creation, assign proper texture.
	minetest.override_item( fencename, {
		tiles = { fencetex }
	})
	-- Previous MtG versions forced a recipe, so delete.
	minetest.clear_craft({output = fencename})

	-- Only add light blue recipe if its mod is loaded.
	if cropo or pcolour ~= "light_blue" then
		minetest.register_craft({
			output = fencename .. " 4",
			recipe = {
				{ 'group:wood', 'group:stick', 'group:wood' },
				{ 'group:wood', 'group:stick', 'group:wood' },
				{ '', dyename, '' },
			}
		})
		-- Mushroom version for ethereal
		if ethshroom ~= "" then
			minetest.register_craft({
				output = fencename .. " 4",
				recipe = {
					{ ethshroom, 'group:stick', ethshroom },
					{ ethshroom, 'group:stick', ethshroom },
					{ '', dyename, '' },
				}
			})
		end
		-- These fences can be repainted.
		minetest.register_craft({
			output = fencename,
			recipe = {
				{ 'group:paint_fence' },
				{ dyename },
			}
		})
	end
	minetest.register_craft({
		type = "fuel",
		recipe = fencename,
		burntime = 7
	})

	-- Create painted rail (not available in 0.4.x)
	if default.register_fence_rail ~= nil then
		default.register_fence_rail(frailname, {
			description = S(fraillabel),
			texture = railtex,
--			material = nil, -- Do NOT add crafting recipe here.
			material = frailname, -- Backward compatibility, no material means crash.
			groups = {choppy = 2, oddly_breakable_by_hand = 2, flammable = 2, paint_rail = 1 },
			sounds = default.node_sound_wood_defaults(),
		})
		-- Previous MtG versions forced a recipe, so delete.
		minetest.clear_craft({output = frailname})

		-- Only add light blue recipe if its mod is loaded.
		if cropo or pcolour ~= "light_blue" then
			minetest.register_craft({
				output = frailname .. " 16",
				recipe = {
					{ 'group:wood', 'group:wood' },
					{ dyename, '' },
					{ 'group:wood', 'group:wood' },
				}
			})
			-- Mushroom version for ethereal
			if ethshroom ~= "" then
				minetest.register_craft({
					output = frailname .. " 16",
					recipe = {
						{ ethshroom, ethshroom },
						{ dyename, '' },
						{ ethshroom, ethshroom },
					}
				})
			end
			-- These rails can be repainted.
			minetest.register_craft({
				output = frailname,
				recipe = {
					{ 'group:paint_rail' },
					{ dyename },
				}
			})
		end
		minetest.register_craft({
			type = "fuel",
			recipe = frailname,
			burntime = 5
		})
	end

	-- doors forces its own recipe, so use painted fence as material for now.
	if gates then
		local gname = "painted_fences:gate_" .. pcolour
		local glabel = plabel .. " Fence Gate"

		doors.register_fencegate(gname, {
			description = S(glabel),
			texture = railtex, -- Gates use rail texture.
			material = fencename,
			groups = {choppy = 2, oddly_breakable_by_hand = 2, flammable = 2,
				paint_gate = 1}
		})
		-- Rip out recipe, replace with own. (Use output, as input is SLOW.)
		minetest.clear_craft({output = gname .. "_closed"})

		-- Only add light blue recipe if its mod is loaded.
		if cropo or pcolour ~= "light_blue" then
			minetest.register_craft({
				output = gname .. "_closed",
				recipe = {
					{"group:stick", "group:wood", "group:stick"},
					{"group:stick", "group:wood", "group:stick"},
					{"", dyename, ""}
				}
			})
			-- Mushroom version for ethereal
			if ethshroom ~= "" then
				minetest.register_craft({
					output = gname .. "_closed",
					recipe = {
						{"group:stick", ethshroom, "group:stick"},
						{"group:stick", ethshroom, "group:stick"},
						{"", dyename, ""}
					}
				})
			end
			-- Gate repainting recipe (for closed node, since that's what drops)
			minetest.register_craft({
				output = gname .. "_closed",
				recipe = {
					{ 'group:paint_gate' },
					{ dyename },
				}
			})
		end
		minetest.register_craft({
			type = "fuel",
			recipe = gname .. "_closed",
			burntime = 7
		})
	end

	if mobsredo then

		local fmobname  = "painted_fences:mobfence_" .. pcolour
		local fmoblabel = plabel .. " Mob Fence"
		local topname   = "painted_fences:mobtop_" .. pcolour
		local toplabel  = plabel .. " Mob Fence Top"

		-- mob fence (looks like normal fence but collision is 2 high)
		default.register_fence(fmobname, {
			description = S(fmoblabel),
			texture = fencetex,
			inventory_image = "mob_fence_overlay.png^" .. fencetex .. "^" ..
				"mob_fence_overlay.png^[makealpha:255,126,126",
--			material = nil, -- I have my own recipe!
			material = fencename, -- Backward compatibility
			groups = {choppy = 2, oddly_breakable_by_hand = 2, flammable = 2,
				paint_mobfence = 1},
			sounds = default.node_sound_wood_defaults(),
			collision_box = {
				type = "fixed",
				fixed = {
					{-0.5, -0.5, -0.5, 0.5, 1.9, 0.5},
				}
			}
		})
		-- Previous MtG versions forced a recipe, so delete.
		minetest.clear_craft({output = fmobname})

		-- Only add light blue recipe if its mod is loaded.
		if cropo or pcolour ~= "light_blue" then
			-- The painted fence can become a painted mob fence and vice versa.
			minetest.register_craft({
				output = fmobname,
				recipe = {
					{ fencename },
				}
			})
			minetest.register_craft({
				output = fencename,
				recipe = {
					{ fmobname },
				}
			})
			-- A standard mob fence can be painted.
			minetest.register_craft({
				output = fmobname,
				recipe = {
					{ 'mobs:fence_wood' },
					{ dyename },
				}
			})
			-- A painted mob fence can be repainted.
			minetest.register_craft({
				output = fmobname,
				recipe = {
					{ 'group:paint_mobfence' },
					{ dyename },
				}
			})
		end
		minetest.register_craft({
			type = "fuel",
			recipe = fmobname,
			burntime = 7
		})

		-- mob fence top (has enlarged collision box to stop mobs getting over)
		minetest.register_node(topname, {
			description = S(toplabel),
			drawtype = "nodebox",
			tiles = { railtex }, -- Fence tops use rail texture.
			paramtype = "light",
			is_ground_content = false,
			groups = {choppy = 2, oddly_breakable_by_hand = 2, flammable = 2, axey = 1,
				paint_mobtop = 1},
			sounds = default.node_sound_wood_defaults(),
			node_box = {
				type = "fixed",
				fixed = {-0.2, -0.5, -0.2, 0.2, 0, 0.2}
			},
			collision_box = {
				type = "fixed",
				fixed = {-0.4, -1.5, -0.4, 0.4, 0, 0.4}
			},
			selection_box = {
				type = "fixed",
				fixed = {-0.4, -1.5, -0.4, 0.4, 0, 0.4}
			}
		})

		-- Only add light blue recipe if its mod is loaded.
		if cropo or pcolour ~= "light_blue" then
			-- Mob fence tops can be made from any wood plus dye.
			minetest.register_craft({
				output = topname .. " 12",
				recipe = {
					{"group:wood", "group:wood", "group:wood"},
					{"", fencename, ""}
				}
			})
			-- Mushroom version for ethereal
			if ethshroom ~= "" then
				minetest.register_craft({
					output = topname .. " 12",
					recipe = {
						{ ethshroom, ethshroom, ethshroom },
						{ "", fencename, "" }
					}
				})
			end
			-- Paint or repaint.
			minetest.register_craft({
				output = topname,
				recipe = {
					{"mobs:fence_top"},
					{dyename}
				}
			})
			minetest.register_craft({
				output = topname,
				recipe = {
					{"group:paint_mobtop"},
					{dyename}
				}
			})
		end
		minetest.register_craft({
			type = "fuel",
			recipe = topname,
			burntime = 2
		})
	end

end


-- Now run this function for the list of colours:
for _, cols in pairs(dyelist) do

    local colour = cols[1]
    local label = cols[2]

    paintem(colour, label)

end

-- Then do the same in a different file:
if gates2 then
	dofile(modpath .. "gates_long.lua")
end

minetest.log("action", "[painted_fences] loaded.")
